#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include <mpi.h>

typedef double real_t;
typedef int64_t int_t;

#define SIZE 4000
#define OUTPUT 500

int_t t = 0, timesteps = SIZE*20;

// MPI partitioning
int_t local_size;   // My chunk of the array, takes the place of 'N'
int rank, nranks; // Communicator rank, size
int left, right;

// Allocate arrays with 1 extra element at each end
// Offset the indexing by 1, so first and last cell
// become "-1" and "N"
real_t *activator[2], *inhibitor[2];
#define A(i) activator[(t)%2][(i)+1]
#define B(i) inhibitor[(t)%2][(i)+1]
#define A_nxt(i) activator[(t+1)%2][(i)+1]
#define B_nxt(i) inhibitor[(t+1)%2][(i)+1]

// Global state at rank 0, for writing to file
real_t *a_output, *b_output;

// Constants for the simulation
#define S (0.02*0.99)
#define RA 0.02
#define DA 0.01
#define BA 0.001
#define DB 0.40
#define RB 0.03


int
main ( int argc, char **argv )
{
    // Split the cell array in equal parts
    // (This split assumes that nranks evenly divides SIZE)
    MPI_Init ( &argc, &argv );
    MPI_Comm_rank ( MPI_COMM_WORLD, &rank );
    MPI_Comm_size ( MPI_COMM_WORLD, &nranks );
    local_size = SIZE / nranks;

    // Set up neighbors
    if ( rank > 0 )
        left = rank-1;
    else
        left = MPI_PROC_NULL;   // Special rank that means "nobody"

    if ( rank < nranks-1 )
        right = rank+1;
    else
        right = MPI_PROC_NULL;

    activator[0] = malloc ( (local_size+2)*sizeof(real_t) );
    activator[1] = malloc ( (local_size+2)*sizeof(real_t) );
    inhibitor[0] = malloc ( (local_size+2)*sizeof(real_t) );
    inhibitor[1] = malloc ( (local_size+2)*sizeof(real_t) );

    if ( rank == 0 )
    {
        a_output = malloc ( SIZE*sizeof(real_t) );
        b_output = malloc ( SIZE*sizeof(real_t) );
    }

    for ( int_t n=0; n<local_size; n++ )
        A(n) = B(n) = 1.0; 

    // Perturb the initial state in a random place
    srand(time(NULL)+rank);
    A(abs(rand())%local_size) = 1.2;

    for ( t=0; t<timesteps; t++ )
    {
        // Boundary condition only at first and last rank
        if ( rank == 0 )
        {
            A_nxt(-1) = A(-1) = A(0);
            B_nxt(-1) = B(-1) = B(0);
        }
        else if ( rank == nranks-1 )
        {
            A_nxt(local_size) = A(local_size) = A(local_size-1);
            B_nxt(local_size) = B(local_size) = B(local_size-1);
        }

        // Exchange the boundary values every step:
        // Send left, receive from right
        MPI_Sendrecv (
            &A(0),          1, MPI_DOUBLE, left, 0,
            &A(local_size), 1, MPI_DOUBLE, right, 0,
            MPI_COMM_WORLD, MPI_STATUS_IGNORE
        );
        MPI_Sendrecv (
            &B(0),          1, MPI_DOUBLE, left, 0,
            &B(local_size), 1, MPI_DOUBLE, right, 0,
            MPI_COMM_WORLD, MPI_STATUS_IGNORE
        );
        // Send right, receive from left
        MPI_Sendrecv (
            &A(local_size-1), 1, MPI_DOUBLE, right, 0,
            &A(-1),           1, MPI_DOUBLE, left, 0,
            MPI_COMM_WORLD, MPI_STATUS_IGNORE
        );
        MPI_Sendrecv (
            &B(local_size-1), 1, MPI_DOUBLE, right, 0,
            &B(-1),           1, MPI_DOUBLE, left, 0,
            MPI_COMM_WORLD, MPI_STATUS_IGNORE
        );

        // Approximate da/dt and db/dt with finite differences, integrate
        for ( int_t n=0; n<local_size; n++ )
        {
            real_t da_dt = DA * ( A(n-1) - 2.0*A(n) + A(n+1) );
            real_t db_dt = DB * ( B(n-1) - 2.0*B(n) + B(n+1) );
            A_nxt(n) = A(n) + (S*A(n)*A(n)+BA) / B(n) - RA * A(n) + da_dt;
            B_nxt(n) = B(n) + S*A(n)*A(n) - RB*B(n) + db_dt;
        }

        // Dump a snapshot of the situation every OUTPUT time steps
        if ( (t%OUTPUT) == 0 )
        {
            // Collect the whole domains at rank 0
            MPI_Gather (
                &A(0), local_size, MPI_DOUBLE,
                a_output, local_size, MPI_DOUBLE,
                0, MPI_COMM_WORLD
            );
            MPI_Gather (
                &B(0), local_size, MPI_DOUBLE,
                b_output, local_size, MPI_DOUBLE,
                0, MPI_COMM_WORLD
            );

            // Rank 0 opens the file and prints everything
            if ( rank == 0 )
            {
                char filename[256];
                memset ( filename, 0, 256*sizeof(char) );
                sprintf ( filename, "data/%.5ld.txt", t );
                FILE *out = fopen ( filename, "w" );
                fprintf ( out, "\"Activator\"\n" );
                for ( int_t n=0; n<SIZE; n++ )
                    fprintf ( out, "%ld %e\n", n, a_output[n] );
                fprintf ( out, "\n\n\"Inhibitor\"\n" );
                for ( int_t n=0; n<SIZE; n++ )
                    fprintf ( out, "%ld %e\n", n, b_output[n] );
                fclose ( out );
            }
        }

    }

    free ( activator[0] ), free ( activator[1] );
    free ( inhibitor[0] ), free ( inhibitor[1] );
    if ( rank == 0 )
    {
        free ( a_output ), free ( b_output );
    }
    MPI_Finalize();
    exit ( EXIT_SUCCESS );
}
