#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>

typedef double real_t;
typedef int64_t int_t;

#define SIZE 4000
#define OUTPUT 500

int_t N = SIZE, t = 0, timesteps = SIZE*20;

real_t *activator[2], *inhibitor[2];
#define A(i) activator[(t)%2][(i)]
#define B(i) inhibitor[(t)%2][(i)]
#define A_nxt(i) activator[(t+1)%2][(i)]
#define B_nxt(i) inhibitor[(t+1)%2][(i)]

// Constants for the simulation
#define S (0.02*0.99)
#define RA 0.02
#define DA 0.01
#define BA 0.001
#define DB 0.40
#define RB 0.03


int
main ( int argc, char **argv )
{
    activator[0] = malloc ( N*sizeof(real_t) );
    activator[1] = malloc ( N*sizeof(real_t) );
    inhibitor[0] = malloc ( N*sizeof(real_t) );
    inhibitor[1] = malloc ( N*sizeof(real_t) );

    for ( int_t n=0; n<N; n++ )
        A(n) = B(n) = 1.0; 

    // Perturb the initial state
    // A(N/3) = 1.2;

    for ( t=0; t<timesteps; t++ )
    {
        // Boundary condition
        A_nxt(0) = A(0) = A(1), A_nxt(N-1) = A(N-1) = A(N-2);
        B_nxt(0) = B(0) = B(1), B_nxt(N-1) = B(N-1) = B(N-2);

        // Approximate da/dt and db/dt with finite differences, integrate
        for ( int_t n=1; n<N-1; n++ )
        {
            real_t da_dt = DA * ( A(n-1) - 2.0*A(n) + A(n+1) );
            real_t db_dt = DB * ( B(n-1) - 2.0*B(n) + B(n+1) );
            A_nxt(n) = A(n) + (S*A(n)*A(n)+BA) / B(n) - RA * A(n) + da_dt;
            B_nxt(n) = B(n) + S*A(n)*A(n) - RB*B(n) + db_dt;
        }

        // Dump a snapshot of the situation every OUTPUT time steps
        if ( (t%OUTPUT) == 0 )
        {
            char filename[256];
            memset ( filename, 0, 256*sizeof(char) );
            sprintf ( filename, "data/%.5ld.txt", t );
            FILE *out = fopen ( filename, "w" );
            fprintf ( out, "\"Activator\"\n" );
            for ( int_t n=0; n<N; n++ )
                fprintf ( out, "%ld %e\n", n, A(n) );
            fprintf ( out, "\n\n\"Inhibitor\"\n" );
            for ( int_t n=0; n<N; n++ )
                fprintf ( out, "%ld %e\n", n, B(n) );
            fclose ( out );
        }
    }

    free ( activator[0] ), free ( activator[1] );
    free ( inhibitor[0] ), free ( inhibitor[1] );
    exit ( EXIT_SUCCESS );
}
