#define _USE_MATH_DEFINES
#include <iostream>
#include <string>
#include <filesystem>
#include <fstream>
#include <bitset>
#include <cassert>
#include <cmath>
#include "arrrgh/arrrgh.hpp"


constexpr int TEXTURES_ENABLED_BIT                      = 0b00000001;
constexpr int NORMALS_ENABLED_BIT                       = 0b00000010;
constexpr int FLEXIBLE_ELEMENT_BIT                      = 0b00000100;
constexpr int CONNECTIVITY_DEPENDENT_FACES_ENABLED_BIT  = 0b00001000;
constexpr int PACKED_ROUND_EDGE_DATA_BIT                = 0b00010000;
constexpr int PACKED_AVERAGE_NORMALS_BIT                = 0b00100000;

enum class RegionType {
    StaticGeometry, ReplaceableGeometry, HideableGeometry
};

struct RegionCondition {
    unsigned int sizeBytes = 0;
    unsigned int isStud = 0;
    unsigned int boolean0 = 0;
    unsigned int options = 0;
    unsigned int boolean1 = 0;
    unsigned int boolean2 = 0;
    unsigned int boolean3 = 0;
};

struct StaticGeometryRegion {
    unsigned int startVertex = 0;
    unsigned int endVertex = 0;
    unsigned int startIndex = 0;
    unsigned int endIndex = 0;
    unsigned int unknown0 = 0; // usually 0
    unsigned int unknown1 = 0; // usually 0
};

struct ReplaceableGeometryRegion {
    unsigned int startVertex = 0;
    unsigned int vertexCount = 0;
    unsigned int startIndex = 0;
    unsigned int indexCount = 0;
    unsigned int sizeOfSorts = 0; // scales linearly with the number of conditional entries
    unsigned int dependsOnConditionals = 0; 

    std::vector<RegionCondition> conditions;

    unsigned int replacementVertexCount = 0;
    unsigned int replacementIndexCount = 0;

    std::vector<float> vertexBuffer;
    std::vector<float> normalBuffer;
    std::vector<float> texCoordBuffer;
    std::vector<unsigned int> indexBuffer;
};

struct HideableGeometryRegion {
    unsigned int startVertex = 0;
    unsigned int vertexCount = 0;
    unsigned int startIndex = 0;
    unsigned int indexCount = 0;
    unsigned int unknown0 = 0; // usually 0
    unsigned int unknown1 = 0; // usually 1

    std::vector<RegionCondition> conditions;
};

struct GBrick {
    // General header options
    bool texturesEnabled = false;
    bool normalsEnabled = false;
    bool isFlexible = false;
    bool connectivityDependentFacesEnabled = false;
    bool packedRoundEdgeDataEnabled = false;
    bool packedAverageNormalsEnabled = false;

    unsigned int vertexCount = 0;
    unsigned int indexCount = 0;

    std::vector<float> vertexBuffer;
    std::vector<float> normalBuffer;
    std::vector<float> texCoordBuffer;
    std::vector<unsigned int> indexBuffer;

    std::vector<StaticGeometryRegion> staticGeometryRegions;
    std::vector<ReplaceableGeometryRegion> replaceableGeometryRegions;
    std::vector<HideableGeometryRegion> hideableGeometryRegions;

    // The file describes geometry regions from start to back.
    // We keep track of their order here so we can easily remove geometry afterwards
    std::vector<RegionType> regionOrder;

    void applyCompactions() {
        unsigned int nextStaticRegionIndex = staticGeometryRegions.size();
        unsigned int nextReplaceableRegionIndex = replaceableGeometryRegions.size();
        unsigned int nextHideableRegionIndex = hideableGeometryRegions.size();

        std::vector<float> addedVertices;
        std::vector<float> addedNormals;
        std::vector<float> addedTexCoords;
        std::vector<unsigned int> addedIndices;

        std::vector<int> markedVertices(vertexBuffer.size() / 3);
        std::vector<int> markedIndices(indexBuffer.size());
        std::vector<int> indexOffsets(vertexBuffer.size() / 3);
        
        for(unsigned int regionIndex = 0; regionIndex < regionOrder.size(); regionIndex++) {
            RegionType regionType = regionOrder.at(regionIndex);
            if(regionType == RegionType::StaticGeometry) {
                // No need to do anything
                assert(nextStaticRegionIndex > 0);
                nextStaticRegionIndex--;
            } else if(regionType == RegionType::HideableGeometry) {
                assert(nextHideableRegionIndex > 0);
                nextHideableRegionIndex--;

                HideableGeometryRegion region = hideableGeometryRegions.at(nextHideableRegionIndex);

                // Mark vertices for future deletion
                for(unsigned int i = region.startVertex; i < region.startVertex + region.vertexCount; i++) {
                    markedVertices.at(i) = 1;
                }

                // Mark indices for future deletion
                for(unsigned int i = region.startIndex; i < region.startIndex + region.indexCount; i++) {
                    markedIndices.at(i) = 1;
                }

                
            } else if(regionType == RegionType::ReplaceableGeometry) {
                assert(nextReplaceableRegionIndex > 0);
                nextReplaceableRegionIndex--;

                ReplaceableGeometryRegion region = replaceableGeometryRegions.at(nextReplaceableRegionIndex);

                // Mark vertices for future deletion
                for(unsigned int i = region.startVertex; i < region.startVertex + region.vertexCount; i++) {
                    markedVertices.at(i) = 1;
                }
                
                // Mark indices for future deletion
                for(unsigned int i = region.startIndex; i < region.startIndex + region.indexCount; i++) {
                    markedIndices.at(i) = 1;
                }

                unsigned int totalAddedVertexCount = addedVertices.size() / 3;
                unsigned int indexBufferStart = addedIndices.size();

                addedVertices.insert(addedVertices.end(), region.vertexBuffer.begin(), region.vertexBuffer.end());
                addedNormals.insert(addedNormals.end(), region.normalBuffer.begin(), region.normalBuffer.end());
                addedTexCoords.insert(addedTexCoords.end(), region.texCoordBuffer.begin(), region.texCoordBuffer.end());
                addedIndices.insert(addedIndices.end(), region.indexBuffer.begin(), region.indexBuffer.end());

                for(unsigned int i = indexBufferStart; i < addedIndices.size(); i++) {
                    // We need to offset the indices, as their definitions inside the file start at 0
                    addedIndices.at(i) += totalAddedVertexCount;
                }
            }
        }

        // Compute index offsets
        int totalOffset = 0;
        for(unsigned int i = 0; i < markedVertices.size(); i++) {
            indexOffsets.at(i) = totalOffset;
            totalOffset += markedVertices.at(i);
        }

        // Compute new index buffer
        for(unsigned int i = 0; i < indexBuffer.size(); i++) {
            unsigned int index = indexBuffer.at(i);
            int indexOffset = markedVertices.at(index);
            index -= indexOffset;
            indexBuffer.at(i) = index;
        }

        // Remove marked indices
        unsigned int targetIndex = 0;
        for(unsigned int sourceIndex = 0; sourceIndex < indexBuffer.size(); sourceIndex++) {
            int mark = markedIndices.at(sourceIndex);
            indexBuffer.at(targetIndex) = indexBuffer.at(sourceIndex);
            targetIndex += 1 - mark;
        }
        indexBuffer.resize(targetIndex);

        // Remove marked vertices
        targetIndex = 0;
        for(unsigned int sourceIndex = 0; sourceIndex < markedVertices.size(); sourceIndex++) {
            int mark = markedVertices.at(sourceIndex);

            vertexBuffer.at(3 * targetIndex + 0) = vertexBuffer.at(3 * sourceIndex + 0);
            vertexBuffer.at(3 * targetIndex + 1) = vertexBuffer.at(3 * sourceIndex + 1);
            vertexBuffer.at(3 * targetIndex + 2) = vertexBuffer.at(3 * sourceIndex + 2);

            normalBuffer.at(3 * targetIndex + 0) = normalBuffer.at(3 * sourceIndex + 0);
            normalBuffer.at(3 * targetIndex + 1) = normalBuffer.at(3 * sourceIndex + 1);
            normalBuffer.at(3 * targetIndex + 2) = normalBuffer.at(3 * sourceIndex + 2);

            if(texturesEnabled) {
                texCoordBuffer.at(2 * targetIndex + 0) = texCoordBuffer.at(2 * sourceIndex + 0);
                texCoordBuffer.at(2 * targetIndex + 1) = texCoordBuffer.at(2 * sourceIndex + 1);
            }

            // Move the target index if vertex was not marked
            targetIndex += 1 - mark;
        }
        // Since we removed the vertices and compacted the buffer, we cut off the remainder to the vertices we removed
        vertexBuffer.resize(3 * targetIndex);
        normalBuffer.resize(3 * targetIndex);
        if(texturesEnabled) {
            texCoordBuffer.resize(2 * targetIndex);
        }

        // Update index buffer
        unsigned int totalVertexCount = vertexBuffer.size() / 3;
        for(unsigned int i = 0; i < addedIndices.size(); i++) {
            addedIndices.at(i) += totalVertexCount;
        }

        // Append new geometry
        vertexBuffer.insert(vertexBuffer.end(), addedVertices.begin(), addedVertices.end());
        normalBuffer.insert(normalBuffer.end(), addedNormals.begin(), addedNormals.end());
        texCoordBuffer.insert(texCoordBuffer.end(), addedTexCoords.begin(), addedTexCoords.end());
        indexBuffer.insert(indexBuffer.end(), addedIndices.begin(), addedIndices.end());
    }

    void dumpToOBJ(std::filesystem::path &outputFile) {
        std::ofstream outFile(outputFile);
        unsigned int vertexCount = vertexBuffer.size() / 3;
         
        for(unsigned int i = 0; i < vertexCount; i++) {
            outFile << "v " << vertexBuffer.at(3 * i + 0) << " "
                            << vertexBuffer.at(3 * i + 1) << " "
                            << vertexBuffer.at(3 * i + 2) << std::endl;
            if(texturesEnabled) {
                outFile << "vt " << texCoordBuffer.at(2 * i + 0) << " "
                        << texCoordBuffer.at(2 * i + 1) << std::endl;
            }
        }
        for (unsigned int i = 0; i < vertexCount; i++) {
            outFile << "vn " << normalBuffer.at(3 * i + 0) << " "
                << normalBuffer.at(3 * i + 1) << " "
                << normalBuffer.at(3 * i + 2) << std::endl;
        }
        if (texturesEnabled) {
            for (unsigned int i = 0; i < vertexCount; i++) {
                outFile << "vt " << texCoordBuffer.at(2 * i + 0) << " "
                    << texCoordBuffer.at(2 * i + 1) << std::endl;
            }
        }
        outFile << std::endl;

        outFile << "g brick" << std::endl;
        
        unsigned int triangleCount = indexBuffer.size() / 3;
        for(unsigned i = 0; i < triangleCount; i++) {
            unsigned int index0 = indexBuffer.at(3 * i + 0) + 1;
            unsigned int index1 = indexBuffer.at(3 * i + 1) + 1;
            unsigned int index2 = indexBuffer.at(3 * i + 2) + 1;

            outFile << "f ";
            
            outFile << index0 << "/";
            if(texturesEnabled) {
                outFile << index0;
            }
            outFile << "/" << index0 << " ";

            outFile << index1 << "/";
            if(texturesEnabled) {
                outFile << index1;
            }
            outFile << "/" << index1 << " ";

            outFile << index2 << "/";
            if(texturesEnabled) {
                outFile << index2;
            }
            outFile << "/" << index2 << std::endl;
        }
    }





};

RegionCondition readConditionalSection(std::ifstream &fileStream) {
    RegionCondition condition;

    fileStream.read((char*) &condition.sizeBytes, sizeof(unsigned int));
    fileStream.read((char*) &condition.isStud, sizeof(unsigned int));
    fileStream.read((char*) &condition.boolean0, sizeof(unsigned int));
    fileStream.read((char*) &condition.options, sizeof(unsigned int));
    fileStream.read((char*) &condition.boolean1, sizeof(unsigned int));
    fileStream.read((char*) &condition.boolean2, sizeof(unsigned int));
    fileStream.read((char*) &condition.boolean3, sizeof(unsigned int));

    assert(condition.sizeBytes == 28);

    return condition;
}

GBrick readGFile(std::filesystem::path inputFile) {
    
    std::ifstream fileStream(inputFile, std::ios::binary);
    size_t fileSize = std::filesystem::file_size(inputFile);
    GBrick brick;

    // File header

    unsigned int magicBytes;
    unsigned int options;

    fileStream.read((char*) &magicBytes, sizeof(unsigned int));
    fileStream.read((char*) &brick.vertexCount, sizeof(unsigned int));
    fileStream.read((char*) &brick.indexCount, sizeof(unsigned int));
    fileStream.read((char*) &options, sizeof(unsigned int));
    
    brick.texturesEnabled = (options & TEXTURES_ENABLED_BIT) == TEXTURES_ENABLED_BIT;
    brick.normalsEnabled = (options & NORMALS_ENABLED_BIT) == NORMALS_ENABLED_BIT;
    brick.isFlexible = (options & FLEXIBLE_ELEMENT_BIT) == FLEXIBLE_ELEMENT_BIT;
    brick.connectivityDependentFacesEnabled = (options & CONNECTIVITY_DEPENDENT_FACES_ENABLED_BIT) == CONNECTIVITY_DEPENDENT_FACES_ENABLED_BIT;
    brick.packedRoundEdgeDataEnabled = (options & PACKED_ROUND_EDGE_DATA_BIT) == PACKED_ROUND_EDGE_DATA_BIT;
    brick.packedAverageNormalsEnabled = (options & PACKED_AVERAGE_NORMALS_BIT) == PACKED_AVERAGE_NORMALS_BIT;

    std::cout << "\tPart info:" << std::endl;
    std::cout << "\t\tVertex count: " << brick.vertexCount << std::endl;
    std::cout << "\t\tIndex count: " << brick.indexCount << std::endl;
    std::cout << "\t\tNormals enabled: " << (brick.normalsEnabled ? "yes" : "no") << std::endl;
    std::cout << "\t\tTextures enabled: " << (brick.texturesEnabled ? "yes" : "no") << std::endl;
    std::cout << "\t\tFlexible element: " << (brick.isFlexible ? "yes" : "no") << std::endl;
    std::cout << "\t\tConnectivity Dependent Faces: " << (brick.connectivityDependentFacesEnabled ? "yes" : "no") << std::endl;
    std::cout << "\t\tPacked Round Edge Data: " << (brick.packedRoundEdgeDataEnabled ? "yes" : "no") << std::endl;
    std::cout << "\t\tPacked Average Normals: " << (brick.packedAverageNormalsEnabled ? "yes" : "no") << std::endl;


    // Reading main geometry buffers

    brick.indexBuffer.resize(brick.indexCount);
    brick.vertexBuffer.resize(3 * brick.vertexCount);
    brick.normalBuffer.resize(3 * brick.vertexCount);

    fileStream.read((char*) brick.vertexBuffer.data(), brick.vertexBuffer.size() * sizeof(float));
    fileStream.read((char*) brick.normalBuffer.data(), brick.normalBuffer.size() * sizeof(float));
    if(brick.texturesEnabled) {
        brick.texCoordBuffer.resize(2 * brick.vertexCount);
        fileStream.read((char*) brick.texCoordBuffer.data(), brick.texCoordBuffer.size() * sizeof(float));
    }
    fileStream.read((char*) brick.indexBuffer.data(), brick.indexBuffer.size() * sizeof(unsigned int));

    if (brick.packedRoundEdgeDataEnabled) {
        unsigned int packedRoundEdgeCount;
        fileStream.read((char*) &packedRoundEdgeCount, sizeof(unsigned int));

        size_t sectionStartPointer = fileStream.tellg();
        size_t roundEdgePointSize = sizeof(float) * packedRoundEdgeCount;
        size_t roundEdgeIndicesSize = sizeof(unsigned int) * brick.indexCount;
        fileStream.seekg(sectionStartPointer + roundEdgePointSize + roundEdgeIndicesSize);
        std::cout << "\tRound Edge Data: " << sectionStartPointer << " + " << roundEdgePointSize << " + " << roundEdgeIndicesSize << " = " << (sectionStartPointer + roundEdgePointSize + roundEdgeIndicesSize) << "/" << fileSize << std::endl;
        assert(sectionStartPointer + roundEdgePointSize + roundEdgeIndicesSize <= fileSize);
    }

    if (brick.packedAverageNormalsEnabled) {
        unsigned int packedRoundEdgeCount;
        fileStream.read((char*)&packedRoundEdgeCount, sizeof(unsigned int));

        size_t sectionStartPointer = fileStream.tellg();
        size_t averageNormalsSize = 3 * sizeof(float) * packedRoundEdgeCount;
        size_t averageNormalIndicesSize = sizeof(unsigned int) * brick.indexCount;
        fileStream.seekg(sectionStartPointer + averageNormalsSize + averageNormalIndicesSize);
        std::cout << "\tPacked Normals: " << sectionStartPointer << " + " << averageNormalsSize << " + " << averageNormalIndicesSize << " = " << (sectionStartPointer + averageNormalsSize + averageNormalIndicesSize) << "/" << fileSize << std::endl;
        assert(sectionStartPointer + averageNormalsSize + averageNormalIndicesSize <= fileSize);
    }

    if (brick.isFlexible) {
        return brick;
    }

    if (brick.connectivityDependentFacesEnabled) {
        // Metadata sections

        unsigned int metadataSectionCount;
        unsigned int metadataSectionsSizeBytes;

        fileStream.read((char*)&metadataSectionCount, sizeof(unsigned int));
        fileStream.read((char*)&metadataSectionsSizeBytes, sizeof(unsigned int));

        std::cout << "\tFile has: " << metadataSectionCount << " metadata sections with a size of " << metadataSectionsSizeBytes << " bytes." << std::endl;

        for (unsigned int section = 0; section < metadataSectionCount; section++) {
            unsigned int sectionSizeBytes;
            unsigned int sectionType;
            size_t sectionStartPointer = fileStream.tellg();
            fileStream.read((char*)&sectionSizeBytes, sizeof(unsigned int));
            fileStream.read((char*)&sectionType, sizeof(unsigned int));

            if (sectionType == 1) {
                // Geometry which can be hidden
                std::cout << "\t\tHideable geometry with no replacement" << std::endl;
                assert(sectionSizeBytes == 68);

                HideableGeometryRegion region;

                fileStream.read((char*)&region.startVertex, sizeof(unsigned int));
                fileStream.read((char*)&region.vertexCount, sizeof(unsigned int));
                fileStream.read((char*)&region.startIndex, sizeof(unsigned int));
                fileStream.read((char*)&region.indexCount, sizeof(unsigned int));
                fileStream.read((char*)&region.unknown0, sizeof(unsigned int));
                fileStream.read((char*)&region.unknown1, sizeof(unsigned int));

                std::cout << "\t\t\tFrom vertex " << region.startVertex << " to " << (region.startVertex + region.vertexCount) << ", from index " << region.startIndex << " to " << (region.startIndex + region.indexCount) << std::endl;
                std::cout << "\t\t\tUnknown values: " << region.unknown0 << ", " << region.unknown1 << std::endl;

                // What conditions need to be met to activate this geometry simplification?
                unsigned int conditionalsSectionSizeBytes;
                unsigned int conditionalCount;

                fileStream.read((char*)&conditionalsSectionSizeBytes, sizeof(unsigned int));
                fileStream.read((char*)&conditionalCount, sizeof(unsigned int));

                //std::cout << "\t\t\tDepends on " << conditionalCount << " connections being present" << std::endl;

                region.conditions.reserve(conditionalCount);
                for (unsigned int conditional = 0; conditional < conditionalCount; conditional++) {
                    region.conditions.push_back(readConditionalSection(fileStream));
                }

                brick.hideableGeometryRegions.push_back(region);
                brick.regionOrder.push_back(RegionType::HideableGeometry);
            }
            else if (sectionType == 2) {
                // Static geometry that is always visible
                std::cout << "\t\tStatic geometry" << std::endl;
                assert(sectionSizeBytes == 32);

                StaticGeometryRegion region;

                fileStream.read((char*)&region.startVertex, sizeof(unsigned int));
                fileStream.read((char*)&region.endVertex, sizeof(unsigned int));
                fileStream.read((char*)&region.startIndex, sizeof(unsigned int));
                fileStream.read((char*)&region.endIndex, sizeof(unsigned int));
                fileStream.read((char*)&region.unknown0, sizeof(unsigned int));
                fileStream.read((char*)&region.unknown1, sizeof(unsigned int));

                brick.staticGeometryRegions.push_back(region);
                brick.regionOrder.push_back(RegionType::StaticGeometry);

                std::cout << "\t\t\tFrom vertex " << region.startVertex << " to " << region.endVertex << ", from index " << region.startIndex << " to " << region.endIndex << std::endl;
                std::cout << "\t\t\tUnknown values: " << region.unknown0 << ", " << region.unknown1 << std::endl;
            }
            else if (sectionType == 4) {
                // Geometry which defined replacement vertices
                std::cout << "\t\tReplacement geometry" << std::endl;

                ReplaceableGeometryRegion region;

                // Which regions in the vertex and index buffers does this affect

                fileStream.read((char*)&region.startVertex, sizeof(unsigned int));
                fileStream.read((char*)&region.vertexCount, sizeof(unsigned int));
                fileStream.read((char*)&region.startIndex, sizeof(unsigned int));
                fileStream.read((char*)&region.indexCount, sizeof(unsigned int));
                fileStream.read((char*)&region.sizeOfSorts, sizeof(unsigned int));
                fileStream.read((char*)&region.dependsOnConditionals, sizeof(unsigned int));

                std::cout << "\t\t\tFrom vertex " << region.startVertex << " to " << (region.startVertex + region.vertexCount) << ", from index " << region.startIndex << " to " << (region.startIndex + region.indexCount) << std::endl;
                std::cout << "\t\t\tConditionals: " << region.sizeOfSorts << ", " << region.dependsOnConditionals << std::endl;

                if (region.dependsOnConditionals == 1) {
                    // What conditions need to be met to activate this geometry simplification?
                    unsigned int conditionalsSectionSizeBytes;
                    unsigned int conditionalCount;

                    fileStream.read((char*)&conditionalsSectionSizeBytes, sizeof(unsigned int));
                    fileStream.read((char*)&conditionalCount, sizeof(unsigned int));

                    std::cout << "\t\t\tDepends on " << conditionalCount << " connections being present" << std::endl;

                    for (unsigned int conditional = 0; conditional < conditionalCount; conditional++) {
                        region.conditions.push_back(readConditionalSection(fileStream));
                    }
                }

                if (region.sizeOfSorts > 0) {
                    // Reading the replacement geometry

                    fileStream.read((char*)&region.replacementVertexCount, sizeof(unsigned int));
                    fileStream.read((char*)&region.replacementIndexCount, sizeof(unsigned int));

                    std::cout << "\t\t\tReplacement geometry: " << region.replacementVertexCount << " vertices and " << region.replacementIndexCount << " indices" << std::endl;

                    region.vertexBuffer.resize(3 * region.replacementVertexCount);
                    fileStream.read((char*)region.vertexBuffer.data(), region.vertexBuffer.size() * sizeof(float));

                    region.normalBuffer.resize(3 * region.replacementVertexCount);
                    fileStream.read((char*)region.normalBuffer.data(), region.normalBuffer.size() * sizeof(float));

                    if (brick.texturesEnabled) {
                        region.texCoordBuffer.resize(2 * region.replacementVertexCount);
                        fileStream.read((char*)region.texCoordBuffer.data(), region.texCoordBuffer.size() * sizeof(float));
                    }

                    region.indexBuffer.resize(region.replacementIndexCount);
                    fileStream.read((char*)region.indexBuffer.data(), region.indexBuffer.size() * sizeof(unsigned int));
                }

                brick.replaceableGeometryRegions.push_back(region);
                brick.regionOrder.push_back(RegionType::ReplaceableGeometry);
            }
            else {
                std::cout << "Unknown section type detected: " << std::bitset<32>(sectionType) << std::endl;
            }

            // Move the file pointer to the start of the next section
            fileStream.seekg(sectionStartPointer + sectionSizeBytes);
        }
    }
    

    // check that file is used up
    size_t filesize = std::filesystem::file_size(inputFile);
    std::vector<char> remainingContents(filesize);
    fileStream.read(remainingContents.data(), filesize);
    size_t bytesRead = fileStream.gcount();
    std::cout << "\tRemaining bytes: " << bytesRead << std::endl;
    remainingContents.resize(bytesRead);
    /*std::ofstream remainder("remainder.bin", std::ios::binary);
    remainder.write(remainingContents.data(), remainingContents.size());
    remainder.close();*/
    //assert(bytesRead == 0);

    return brick;
}

int main(int argc, const char** argv) {
    arrrgh::parser parser("gconvert", "Convert G files to simplified OBJ files");
    const auto& inputDirectoryArg = parser.add<std::string>("input-directory", "Directory containing .g files.", '\0', arrrgh::Required, "");
    const auto& outputDirectoryArg = parser.add<std::string>("output-directory", "Directory the converted .obj files should be written to.", '\0', arrrgh::Required, "");
    const auto& showHelp = parser.add<bool>("help", "Show this help message.", 'h', arrrgh::Optional, false);

    try
    {
        parser.parse(argc, argv);
    }
    catch (const std::exception& e)
    {
        std::cout << "Error parsing arguments: " << e.what() << std::endl;
        parser.show_usage(std::cout);
        exit(1);
    }

    // Show help if desired
    if(showHelp.value())
    {
        return 0;
    }



    std::filesystem::path inputDirectory (inputDirectoryArg.value());
    std::filesystem::path outputDirectory (outputDirectoryArg.value());

    for (const std::filesystem::directory_entry& dir_entry : std::filesystem::directory_iterator{inputDirectory}) 
    {
        std::string filename = dir_entry.path().filename().string();
        std::cout << "Processing file " << filename << std::endl;
        filename.back() = 'o';
        filename += "bj";
        std::filesystem::path outputFile = outputDirectory / filename;
        GBrick brick = readGFile(dir_entry);
        brick.applyCompactions();
        brick.dumpToOBJ(outputFile);

    }
    std::cout << "Done." << std::endl;
}   