#include <iostream>
#include <string>
#include <filesystem>
#include <fstream>
#include <bitset>
#include <iomanip>
#include <sstream>
#include <cassert>
#include <algorithm>
#include "arrrgh/arrrgh.hpp"
#include "fast_obj.h"

struct vec3 {
    float x = 0;
    float y = 0;
    float z = 0;
};

std::ostream& operator<<(std::ostream& stream, vec3 vector) {
    stream << "[" << vector.x << ", " << vector.y << ", " << vector.z << "]";
    return stream;
}

constexpr int TEXTURES_ENABLED_BIT = 0b00000001;
constexpr int NORMALS_ENABLED_BIT = 0b00000010;
constexpr int FLEXIBLE_ELEMENT_BIT = 0b00000100;
constexpr int CONNECTIVITY_DEPENDENT_FACES_ENABLED_BIT = 0b00001000;
constexpr int PACKED_ROUND_EDGE_DATA_BIT = 0b00010000;
constexpr int PACKED_AVERAGE_NORMALS_BIT = 0b00100000;


void convertOBJToG(std::filesystem::path &inputFile, std::filesystem::path& outputFile) {

    std::ifstream inputFileStream{inputFile.string()};
    std::string line;
    std::cout << "\tReading OBJ file: " << inputFile.string() << std::endl;
    fastObjMesh* temporaryMesh = fast_obj_read(inputFile.string().c_str());

    std::cout << "\tConverting geometry buffers.." << std::endl;
    std::vector<vec3> vertices;
    std::vector<vec3> normals;

    unsigned int faceCount = 0;

    for(unsigned int groupIndex = 0; groupIndex < temporaryMesh->group_count; groupIndex++) {
        faceCount += temporaryMesh->groups[groupIndex].face_count;
    }

    std::wcout << "\t\tObject has " << faceCount << " faces" << std::endl;

    bool hasNormals = temporaryMesh->normal_count != 1;

    vertices.reserve(3 * faceCount);
    normals.reserve(3 * faceCount);

    for(unsigned int groupIndex = 0; groupIndex < temporaryMesh->group_count; groupIndex++) {
        fastObjGroup group = temporaryMesh->groups[groupIndex];

        for (unsigned int faceIndex = 0; faceIndex < group.face_count; faceIndex++) {
            // Ensure faces are triangles. Can probably fix this at a later date, but for now it gives an error.
            unsigned int verticesPerFace = temporaryMesh->face_vertices[faceIndex + group.face_offset];
            if (verticesPerFace != 3) {
                throw std::runtime_error(
                    "This OBJ loader only supports 3 vertices per face. The model file " + inputFile.string() +
                    " contains a face with " + std::to_string(verticesPerFace) + " vertices.\n"
                    "You can usually solve this problem by re-exporting the object from a 3D model editor, "
                    "and selecting the OBJ export option that forces triangle faces.");
            }

            for (unsigned int i = 0; i < 3; i++) {
                fastObjIndex index = temporaryMesh->indices[3 * faceIndex + i + group.index_offset];

                vertices.push_back({
                    temporaryMesh->positions[3 * index.p + 0],
                    temporaryMesh->positions[3 * index.p + 1],
                    temporaryMesh->positions[3 * index.p + 2]});
                

                if (hasNormals) {
                    normals.push_back( {
                        temporaryMesh->normals[3 * index.n + 0],
                        temporaryMesh->normals[3 * index.n + 1],
                        temporaryMesh->normals[3 * index.n + 2]});
                }
            }
        }
    }

    std::wcout << "\t\tVertex Count: " << vertices.size() << std::endl;

    std::vector<unsigned int> indices(3 * faceCount);
    for (unsigned int i = 0; i < 3 * faceCount; i++) {
        indices.at(i) = i;
    }
    std::cout << "\t\Index count: " << (3 * faceCount) << std::endl;

    fast_obj_destroy(temporaryMesh);

    std::cout << "\tWriting .g file.." << std::endl;

    std::ofstream outputFileStream{ outputFile.string() };
    
    const char magicHeader[5] = "10GB";
    outputFileStream.write(magicHeader, 4);

    unsigned int vertexCount = 3 * faceCount;
    outputFileStream.write((const char*) &vertexCount, sizeof(unsigned int));
    outputFileStream.write((const char*) &vertexCount, sizeof(unsigned int));
    unsigned int options = (hasNormals ? NORMALS_ENABLED_BIT : 0);
    outputFileStream.write((const char*)&options, sizeof(unsigned int));

    outputFileStream.write((const char*) vertices.data(), vertexCount * sizeof(vec3));
    if (hasNormals) {
        outputFileStream.write((const char*) normals.data(), vertexCount * sizeof(vec3));
    }
    outputFileStream.write((const char*) indices.data(), indices.size() * sizeof(unsigned int));
}

int main(int argc, const char** argv) {
    const std::string defaultValue = "NONE_SPECIFIED";
    arrrgh::parser parser("gconvert", "Convert G files to simplified OBJ files");
    const auto& inputDirectoryArg = parser.add<std::string>("input-directory", "Directory containing .obj files. Must have an accompanying output directory specified.", '\0', arrrgh::Optional, defaultValue);
    const auto& outputDirectoryArg = parser.add<std::string>("output-directory", "Directory the converted .g files should be written to.", '\0', arrrgh::Optional, defaultValue);
    const auto& inputFileArg = parser.add<std::string>("input-file", "The .obj file that should be converted into a .g file. Must have an accompanying output file.", '\0', arrrgh::Optional, defaultValue);
    const auto& outputFileArg = parser.add<std::string>("output-file", "Where the converted .g file should be written to.", '\0', arrrgh::Optional, defaultValue);
    const auto& showHelp = parser.add<bool>("help", "Show this help message.", 'h', arrrgh::Optional, false);

    try
    {
        parser.parse(argc, argv);
    }
    catch (const std::exception& e)
    {
        std::cout << "Error parsing arguments: " << e.what() << std::endl;
        parser.show_usage(std::cout);
        exit(1);
    }

    // Show help if desired
    if(showHelp.value())
    {
        return 0;
    }

    if (inputDirectoryArg.value() != defaultValue) {
        if (outputDirectoryArg.value() == defaultValue) {
            std::cout << "A value was specified for --input-directory, but no corresponding --output-directory was given. Please specify the output directory!" << std::endl;
            return 0;
        }

        std::filesystem::path inputDirectory(inputDirectoryArg.value());
        std::filesystem::path outputDirectory(outputDirectoryArg.value());

        for (const std::filesystem::directory_entry& dir_entry : std::filesystem::directory_iterator{ inputDirectory })
        {
            std::filesystem::path inputFile = dir_entry.path();
            std::string filename = inputFile.filename().string();
            std::cout << "Processing file " << filename << std::endl;
            filename = filename.substr(0, filename.size() - 4) + ".g";
            std::filesystem::path outputFile = outputDirectory / filename;
            convertOBJToG(inputFile, outputFile);
        }
    }

    if (inputFileArg.value() != defaultValue) {
        if (outputFileArg.value() == defaultValue) {
            std::cout << "A value was specified for --input-file, but no corresponding --output-file was given. Please specify where to write the output file!" << std::endl;
            return 0;
        }

        std::filesystem::path inputFile{ inputFileArg.value() };
        std::filesystem::path outputFile{ outputFileArg.value() };

        convertOBJToG(inputFile, outputFile);
    }

    
    std::cout << "Done." << std::endl;
}   