// Documetation can be found at http://tdt4102.pages.stud.idi.ntnu.no/graph_lib/
/** 
 * @file 
 * @brief A library for Graphical User interface.
 * 
 * This is a GUI support code to the chapters 12-16 of the book
 * "Programming -- Principles and Practice Using C++" by Bjarne Stroustrup
*/
#pragma once

#include "Window.h"
#include "Graph.h"

namespace Graph_lib {

//------------------------------------------------------------------------------
/** @brief Address is a synonym for void* */
using Address = void*; 

/** @brief FLTK's required function type for all callbacks, Callback is a pointer to a function */
using Callback = void(*)(Address, Address);


//------------------------------------------------------------------------------
/** @brief Treat an address as a reference to a W */
template <class W> W& reference_to(Address pw)
{
	return *static_cast<W*>(pw);
}

//------------------------------------------------------------------------------
/**
 * @brief Widget describes a form of interaction through the GUI.
 * 
 * Widget is a handle to an Fl_widget - it is NOT an Fl-widget as we try to keep our interface classes away from FLTK
 */
class Widget
{
public:
	/**
	 * @brief Widget constructor
	 * @param xy Top left corner of widget
	 * @param w Width 
	 * @param h Height
	 * @param s Label of Widget
	 * @param cb Callback function to be called when user interacts with Widget
	 */
	Widget(Point xy, int w, int h, const string& s, Callback cb)
		: loc(xy), width(w), height(h), label(s), do_it(cb)
	{}

	/** @brief Schedules redrawing of widget, as per FLTK does */
	void redraw()
	{
		pw->redraw();
	}

	/** @brief Move widget @p dx in x-direction and @p dy in y-direction */
	virtual void move(int dx, int dy)
	{
		hide();
		pw->position(loc.x += dx, loc.y += dy);
		show();
	}

	/** @brief Hide the widget, it will not be displayed */
	virtual void hide() { pw->hide(); }

	/** @brief Display the widget, a Widget starts out visible */
	virtual void show() { pw->show(); }

	/** @brief Pure virtual for attaching the widget. All widgets must be attached to a window in order to be displayed! */
	virtual void attach(Window&) = 0;

	virtual void clear_value() {}

	/** @brief Change Widget's label to @p l */
	virtual void set_label(string l)
	{
		label = l;
		pw->label(label.c_str());
	}

	Point loc;
	int width;
	int height;
	string label;
	Callback do_it;

	virtual ~Widget()
	{}

protected:
	/** @brief The Window the Widget belongs to, every Widget must belong to one */
	Window* own;

	/** @brief Connection to the FLTK Widget */
	Fl_Widget* pw;
private:
	Widget& operator=(const Widget&); // don't copy Widgets
	Widget(const Widget&);
};

//------------------------------------------------------------------------------
/** @brief Button is the simplest Widget, it calls @c cb when clicked */
struct Button : Widget
{
	/** @brief Button constructor, uses Widget constructor with same parameters. @p label will be placed on Button */
	Button(Point xy, int w, int h, const string& label, Callback cb) : Widget(xy, w, h, label, cb) {}

	/** @brief Attaches the button to its window and displays it */
	void attach(Window&);
};

//------------------------------------------------------------------------------
/** @brief A Widget without a Callback function */ 
struct Choice : Widget
{
	/** @brief Choice constructor, calls Widget's constructor without callback */
	Choice(Point xy, int w, int h, const string& s) : Widget(xy, w, h, s, 0)
	{}

	/** @brief @p item will be written on Choice */
	int add(const string& item);

	/** @brief Removes value of Choice */
	void clear_value() override
	{
		static_cast<Fl_Choice*>(pw)->value(0);
	}
	/** @brief Attach Choice to @p win */
	void attach(Graph_lib::Window& win) override;

	/** @brief Returns the value */
	int value()
	{
		return static_cast<Fl_Choice*>(pw)->value();
	}
};

//------------------------------------------------------------------------------
/** @brief In_box is a Widget where the user can input information */
struct In_box : Widget
{
	/** @brief In_box constructor, calls Widget constructor without callback function */
	In_box(Point xy, int w, int h, const string& s) : Widget(xy, w, h, s, nullptr)
	{}

	/** @brief Returns the input in the In_box provided by user, use if int expected */
	int get_int();

	/** @brief Returns the input in the In_box provided by user, use if string expected */
	string get_string();

	/** @brief Empties the In_box */
	void clear_value() override { static_cast<Fl_Input*>(pw)->value(""); }

	/** @brief Attaches In_box to @p win and displays it */
	void attach(Window& win) override;
};

//------------------------------------------------------------------------------
/** @brief Out_box is a Widget where the program can output messages to screen */
struct Out_box : Widget
{
	/** @brief Out_box constructor, ucalls Widget constructor without callback function */
	Out_box(Point xy, int w, int h, const string& s) : Widget(xy, w, h, s, nullptr)
	{}

	/** @brief Display an int in the Out_box */
	void put(int);

	/** @brief Display a string in the Out_box */
	void put(const string&);

	/** @brief Attach to @p win and display it */
	void attach(Graph_lib::Window& win) override;
};

//------------------------------------------------------------------------------
/** @brief Similar to Out_box, but supports multiple lines. Use the newline character @c \n to break the lines */
struct Multiline_out_box : Widget
{
	/** @brief Constructor, calls Widget's constructor without callback function */
	Multiline_out_box(Point xy, int w, int h, const string& s) : Widget(xy, w, h, s, nullptr)
	{}

	/** @brief Display an integer in the Multiline_out_box */
	void put(int);

	/** @brief Display a string, separate lines with newline character, @c \n */
	void put(const string&);

	/** @brief Attach to the window and display it */
	void attach(Window& win);
};

//------------------------------------------------------------------------------
/** @brief Menu is a collection of multiple Button objects */
struct Menu : Widget
{
	/** @brief Two types of arrangement for the menu, horizontal puts the Buttons next to eachother, vertical puts them below eachother */
	enum Kind { horizontal, vertical };

	/**
	 * @brief Menu constructor, calls Widget constructor without callback function. Sets offset to 0
	 * @param xy Top left corner of xy
	 * @param w Width of entire menu
	 * @param h Height of entire menu
	 * @param kk Type of menu, horizontal or vertical
	 * @param label Label at top of menu
	 */
	Menu(Point xy, int w, int h, Kind kk, const string& label)
		: Widget(xy, w, h, label, nullptr), k(kk), offset(0) {}

	/** @brief A vector_ref of the Buttons in the menu */
	Vector_ref<Button> selection;

	/** @brief Type of menu, horizontal or vertical */
	Kind k;

	/** @brief Offset between each Button, set to 0 by constructor */
	int offset;

	/**
	 * @brief Attach reference to @p b to the menu.
	 * Menu does not delete @p &b when it goes out of scope
	 */
	int attach(Button& b); // Menu does not delete &b

	/**
	 * @brief Attach pointer @p p to the menu.
	 * Menu deletes p when it goes out of scope
	 */
	int attach(Button* p); // Menu deletes p

	/** @brief Show all buttons */
	void show() 
	{
		for (int i = 0; i < selection.size(); ++i)
			selection[i].show();
	}

	/** @brief Hide all buttons */
	void hide() 
	{
		for (int i = 0; i < selection.size(); ++i)
			selection[i].hide();
	}

	/** @brief Move all buttons */
	void move(int dx, int dy) 
	{
		for (int i = 0; i < selection.size(); ++i)
			selection[i].move(dx, dy);
	}

	/** @brief Attach all buttons to the @p win and display them */
	void attach(Window& win) // attach all buttons
	{
		for (int i = 0; i < selection.size(); ++i)
			win.attach(selection[i]);
		own = &win;
	}

private:
	/** @brief Heavy lifting of attaching button to menu, do not use this */
	void _attach(Button& b); 
};

//------------------------------------------------------------------------------

} // namespace Graph_lib
