// Documetation can be found at http://tdt4102.pages.stud.idi.ntnu.no/graph_lib/
/** 
 @file 
 @brief A library for basic graphic shapes and other attributes
  */
#pragma once

#include "Point.h"
#include "fltk.h"
#include "std_lib_facilities.h"

namespace Graph_lib {
// defense against ill-behaved Linux macros:
#undef major
#undef minor

/**
 @brief A struct to set color and visibility of objects in graphics
  */
struct Color
{
	/** @brief enum of FL colors  */
	enum Color_type {
		red = FL_RED,
		blue = FL_BLUE,
		green = FL_GREEN,
		yellow = FL_YELLOW,
		white = FL_WHITE,
		black = FL_BLACK,
		magenta = FL_MAGENTA,
		cyan = FL_CYAN,
		dark_red = FL_DARK_RED,
		dark_green = FL_DARK_GREEN,
		dark_yellow = FL_DARK_YELLOW,
		dark_blue = FL_DARK_BLUE,
		dark_magenta = FL_DARK_MAGENTA,
		dark_cyan = FL_DARK_CYAN,
		gray = FL_GRAY,
		mid_gray = 48,
		dark_gray = 38,
		light_gray = 50
	};

	/** @brief enum for transparency of a given Color, visible or invisible  */
	enum Transparency { invisible = 0, visible = 255 };
	
	/**
     @brief constructs a Color, will  set visibility to visible
     @param cc desired color, must be contained in the enum @c Color_type
      */
	Color(Color_type cc) : v(visible), c(Fl_Color(cc)) {}
	
	/**
     @brief constructor for Color
     @param cc desired color, must be contained in the enum @c Color_type
	 @param vv transparency, visible or invisible
      */
	Color(Color_type cc, Transparency vv) : v(vv), c(Fl_Color(cc)) {}
	
	/**
     @brief constructor for a Color, sets color based on an integer corresponding to a color in the enum @c Fl_color
     @param cc an integer between 0-255, will set the color according to the enueration type Fl_color
      */
	Color(int cc) : v(visible), c(Fl_Color(cc)) {}
	
	/**
     @brief constructor for Color, sets color to default color and transparency based on @p vv
     @param vv transparency of color, invisible or visible
      */
	Color(Transparency vv) : v(vv), c(Fl_Color()) {}

	/** @brief returns color as an integer  */
	int as_int() const { return c; }
	
	/** @brief returns visibility, 0 or 1  */
	char visibility() const { return v; }
	
	/** @brief used to set visibility, invisible or visible  */
	void set_visibility(Transparency vv) { v = vv; }

private:
	/** @brief visibility of color, 0 or 1  */
	unsigned char v; // 0 or 1 for now

	/** @brief Color  */
	Fl_Color c;
};

/**
 @brief Line_style is a struct for setting styles of lines in graphics
  */
struct Line_style
{
	/** @brief enum of different line styles  */
	enum Line_style_type {
		solid = FL_SOLID,		   // -------
		dash = FL_DASH,			   // - - - -
		dot = FL_DOT,			   // .......
		dashdot = FL_DASHDOT,	   // - . - .
		dashdotdot = FL_DASHDOTDOT // -..-..
	};

	/**
     @brief constructor for Line_style, sets width to default
     @param ss desired style of line, must be contained in the enum @c Line_style_type
      */
	Line_style(Line_style_type ss) : s(ss), w(0) {}

	/**
     @brief constructor for Line_style
     @param lst desired style of line, must be contained in the enum @c Line_style_type
     @param ww desired width of line, in pixels
      */
	Line_style(Line_style_type lst, int ww) : s(lst), w(ww) {}

	/**
     @brief constructor for Line_style, sets width to default width
     @param ss an integer to set the style of the line, between 0 - 4
      */
	Line_style(int ss) : s(ss), w(0) {}

	/** @brief returns the width */
	int width() const { return w; }

	/** @brief returns the style as an int  */
	int style() const { return s; }

private:
	/** @brief Style, contained in the enum  */
	int s;

	/** @brief width of line  */
	int w;
};

/** @brief The class Font is used to set the font of a @c Text ogject in graphics  */
class Font
{
public:
	/** @brief An enum containing the different fonts  */
	enum Font_type {
		helvetica = FL_HELVETICA,
		helvetica_bold = FL_HELVETICA_BOLD,
		helvetica_italic = FL_HELVETICA_ITALIC,
		helvetica_bold_italic = FL_HELVETICA_BOLD_ITALIC,
		courier = FL_COURIER,
		courier_bold = FL_COURIER_BOLD,
		courier_italic = FL_COURIER_ITALIC,
		courier_bold_italic = FL_COURIER_BOLD_ITALIC,
		times = FL_TIMES,
		times_bold = FL_TIMES_BOLD,
		times_italic = FL_TIMES_ITALIC,
		times_bold_italic = FL_TIMES_BOLD_ITALIC,
		symbol = FL_SYMBOL,
		screen = FL_SCREEN,
		screen_bold = FL_SCREEN_BOLD,
		zapf_dingbats = FL_ZAPF_DINGBATS
	};

	/**
     @brief constructor for Font base on a @c Font_type
     @param ff Desired fonttype, must be contained in the enum @c Font_type
      */
	Font(Font_type ff) : f(ff) {}
	
	 /**
     @brief Constructor for Font based on an int
     @param ff desired fonttype described as an int between 0 - 15
      */
	Font(int ff) : f(ff) {}

	/** @brief returns Font type as an integer  */
	int as_int() const { return f; }

private:
	/** @brief font type saved as an int  */
	int f;
};

/** @brief Template class Vector_ref is a special vector for holding unnamed objects of any type  */
template <class T> class Vector_ref
{
	/** @brief Vector with pointers to all the elements in the vector_ref instance  */
	vector<T*> v;

	/** @brief Vector with pointers to all the element in the instance which memory this instance owns  */
	vector<T*> owned;

public:
	/** @brief empty default constructor  */
	Vector_ref() {}

	/** @brief Constructor with up to 4 initial elements to add */
	Vector_ref(T* a, T* b = 0, T* c = 0, T* d = 0)
	{
		if (a)
			push_back(a);
		if (b)
			push_back(b);
		if (c)
			push_back(c);
		if (d)
			push_back(d);
	}

	/** @brief Destructor, deletes all memory it owns  */
	~Vector_ref()
	{
		for (typename vector<T*>::size_type i = 0; i < owned.size(); ++i)
			delete owned[i];
	}

	/** @brief adds @p s at the back of the vector_ref, the instance will not own this memory and user is responsible for deleting this themselves */
	void push_back(T& s) { v.push_back(&s); }
	
	/** @brief adds @p p at the back of the vector_ref */
	void push_back(T* p)
	{
		v.push_back(p);
		owned.push_back(p);
	}

	/**
	 * @brief overloading the []-operator for vector_ref, works the same way as for a normal vector
	 *  returns the value of elemnt number i
	  */
	T& operator[](int i) { return *v[i]; }

	/**
	 * @brief overloading the []-operator for a vector_ref with const elements, see other overload
	  */
	const T& operator[](int i) const { return *v[i]; }

	/** @brief returns number of elements in the vector_ref  */
	int size() const { return v.size(); }
	
	/** @brief returns the last element of the vector_ref  */
	T& back() { return *v[v.size() - 1]; }
	
	/** @brief returns the last element of a vector_ref with const elements  */
	const T& back() const { return *v[v.size() - 1]; }

	using iterator = typename vector<T*>::iterator;

	/** @brief returns an iterator to the first element of vector_ref  */
	iterator begin() { return v.begin(); }

	/** @brief returns an iterator to one element after the last  */
	auto end() { return v.end(); }

	auto begin() const { return v.begin(); }
	auto end() const { return v.end(); }
};

/** @brief Fct is a function with a double parameter and double return type  */
using Fct = double(double);

/** @brief A class which holds sequences of lines and deals with color and style  */

class Shape
{ // deals with color and style, and holds sequence of lines
protected:
	/**
	 * @brief Empty default constructor */
	Shape()
	{}
	// Not by the book:
	// Shape(initializer_list<Point> lst); // add() the Points to this Shape
	// Changed init_list constructor, i.e. implemented at all
	/**
	 * @brief A constructor for shape, will create a shape by adding the elements of @p lst to @c points
	 * @param lst a list of points to be added to the shape
	 * This constructor is not as described by the book
	  */
	Shape(initializer_list<Point> lst)
	{
		for (Point p : lst)
			add(p);
	} // add() the Points to this Shape

	/** @brief adds a Point to @c points of the Shape  */
	void add(Point p) { points.push_back(p); }

	/** @brief changes the position of element @p i in @c points to @p p  */
	void set_point(int i, Point p) { points[i] = p; }

public:
	/** @brief deal with color and draw_lines  */
	void draw() const; // deal with color and draw_lines
protected:
	/** @brief simply draw the appropriate lines  */
	virtual void draw_lines() const; // simply draw the appropriate lines
public:
	/** @brief move the shape +=dx and +=dy  */
	virtual void move(int dx, int dy); // move the shape +=dx and +=dy

	/** @brief Set the color of the shape's lines to @p col  */
	void set_color(Color col) { lcolor = col; }

	/** @brief Returns the color of the shape's lines */
	Color color() const { return lcolor; }

	/** @brief Set the style of the shape's lines to the @c Line_style @p sty  */
	void set_style(Line_style sty) { ls = sty; }

	/** @brief Returns the style of the shape's lines  */
	Line_style style() const { return ls; }

	/** @brief Set the color of the shape's inside to @p col  */
	void set_fill_color(Color col) { fcolor = col; }

	/** @brief Returns the color of the shape's fill  */
	Color fill_color() const { return fcolor; }

	/** @brief Returns point number @p i in @c points  */
	Point point(int i) const { return points[i]; }

	/** @brief Returns number of points/corners in the shape  */
	int number_of_points() const { return int(points.size()); }

	/** @brief Destructor for Shape, empty  */
	virtual ~Shape()
	{}

	/** @brief Deleted copyconstructor  */
	Shape(const Shape&) = delete;

	/** @brief Deleted assignmentoperator  */
	Shape& operator=(const Shape&) = delete;

private:
/** @brief a vector of all the Points in the shape, lines will be drawn between these, not used by all shapes */
	vector<Point> points; // not used by all shapes

	/** @brief Color of the lines of the shape */
	Color lcolor{static_cast<int>(fl_color())};

	/** @brief Style of the lines of the shape */
	Line_style ls{0};

	/** @brief color of the inside/fill of the shape */
	Color fcolor{Color::invisible};
};

/** @brief A function, inherits from shape */
struct Function : Shape
{
	// the function parameters are not stored
	/**
	 * @brief Function constructor
	 * @param f f(x), function, must be @c Fct object
	 * @param r1 startvalue of x
	 * @param r2 endvalue of x
	 * @param origo (0,0)
	 * @param count number of points to calculate within the range
	 * @param xscale scale to scale x-coordinates by
	 * @param yscale scale to scale y-coordinates by
	 *  the function parameters are not stored, only used in initialization
	  */
	Function(Fct f, double r1, double r2, Point orig, int count = 100, double xscale = 25,
			 double yscale = 25);
};

/** @brief Fill to use in Shapes */
struct Fill
{
	Fill() : no_fill(true), fcolor(0)
	{}
	Fill(Color c) : no_fill(false), fcolor(c)
	{}

	void set_fill_color(Color col) { fcolor = col; }
	Color fill_color() { return fcolor; }

protected:
	bool no_fill;
	Color fcolor;
};

/** @brief Line is a subclass of Shape with only two Points and a line between them */
struct Line : Shape
{
	/** @brief Constructor for a Line between @p p1 and @p p2 */
	Line(Point p1, Point p2)
	{
		add(p1);
		add(p2);
	}
};

/** @brief Rectangle is a subclass of Shape with four points */
struct Rectangle : Shape
{
	/**
	 * @brief Basic constructor for a Rectangle given one Point, checks that arguments will make a valid rectangle, throws exception if not
	 * @param xy top left corner of rectangle
	 * @param ww width of rectangle
	 * @param height of rectangle
	  */
	Rectangle(Point xy, int ww, int hh) : h{hh}, w{ww}
	{
		if (h <= 0 || w <= 0)
			error("Bad rectangle: non-positive side");
		add(xy);
	}
	/**
	 * @brief Basic constructor for a Rectangle given two Points, checks that arguments make a valid rectangle, throws exception if not
	 * @param x top left corner
	 * @param y bottom right corner
	  */ 
	Rectangle(Point x, Point y) : h{y.y - x.y}, w{y.x - x.x}
	{
		if (h <= 0 || w <= 0)
			error("Bad rectangle: first point is not top left");
		add(x);
	}

	/** @brief Draws the rectangle */
	void draw_lines() const;

	/** @brief returns height of rectangle */
	int height() const { return h; }

	/** @brief returns width of rectangle */
	int width() const { return w; }

private:
	/** @brief height */
	int h;

	/** @brief width */
	int w; 
};

/** @brief True if lines between the points will intersect, False otherwise */
bool intersect(Point p1, Point p2, Point p3, Point p4);

/** @brief An open sequence of lines, inherits from Shape */
struct Open_polyline : Shape
{ 
	using Shape::Shape;

	/** @brief adds @p p to the list of points in the line */
	void add(Point p) { Shape::add(p); }

	/** @brief draws lines between the points, does not connect last to first */
	void draw_lines() const;
};

/** @brief A closed sequence of lines, inherits from Open_polyline */
struct Closed_polyline : Open_polyline
{ 
	using Open_polyline::Open_polyline;

	/** @brief draws lines between the points, connects first to last */
	void draw_lines() const;

	//	void add(Point p) { Shape::add(p); }
};

/** @brief A closed sequence of non-intersecting lines, inherits from Closed_polyline */
struct Polygon : Closed_polyline
{ 
	using Closed_polyline::Closed_polyline;

	/** @brief adds the Point @p p to the sequence of lines, throws an error if the new @p p violates the demands of a Polygon */
	void add(Point p);

	/** @brief draws lines between all Points connects last to first, throws an error if there are less than three Points */
	void draw_lines() const;
};

/** @brief Independent lines, inherits from Shape */
struct Lines : Shape
{
	/** @brief Empty default constructor */
	Lines()
	{}
	/**
	 * @brief Constructor for Lines, uses Shape constructor but checks for equal number of Points
	 * @param lst a list of Points
	 *  Not equal to the one given by the book
	  */
	Lines(initializer_list<Point> lst) : Shape{lst}
	// Not equal to book constructor.
	{
		if (lst.size() % 2)
			error("odd number of points for Lines");
	}

	/**
	 * @brief Constructor for Lines, adds the Points in pairs
	 * @param lst a list of @c Pair of Points
	 *  This is the constructor as described by the book
	  */
	Lines(initializer_list<pair<Point, Point>> lst)
	{
		// This is the constructor from the book, p.450 1st print.
		for (auto p : lst)
			add(p.first, p.second);
	}

	/** @brief draws lines between two and two points  */
	void draw_lines() const;

	/** @brief adds two Points to the member-vector @c points  */
	void add(Point p1, Point p2)
	{
		Shape::add(p1);
		Shape::add(p2);
	}
};

/** @brief A text struct, inherits from Shape  */
struct Text : Shape
{
	/**
	 * @brief Constructor for Text
	 * @param x the point is the bottom left of the first letter
	 * @param s the tezt to be displayed
	 */
	Text(Point x, const string& s) : lab{s} { add(x); }

	/** @brief Draws the text with given font, size and linestyle  */
	void draw_lines() const;

	/** @brief change the text to @p s  */
	void set_label(const string& s) { lab = s; }

	/** @brief returns the text as a string  */
	string label() const { return lab; }

	/** @brief set the font of the text to @p f  */
	void set_font(Font f) { fnt = f; }

	/** @brief returns the current font of the text  */
	Font font() const { return Font(fnt); }

	/** @brief set the size of the text to @p s at least 14  */
	void set_font_size(int s) { fnt_sz = s; }

	/** @brief returns the size of the font  */
	int font_size() const { return fnt_sz; }

private:
	/** @brief label/text  */
	string lab;

	/** @brief font  */
	Font fnt{fl_font()};

	/** @brief size of font, at least 14 points  */
	int fnt_sz{(14 < fl_size()) ? fl_size() : 14}; // at least 14 point
};

/** @brief A struct for simple representation of axes in graphics, inherits from Shape  */
struct Axis : Shape
{
	// representation left public
	/** @brief enum containing the three possible orientations of an Axis  */
	enum Orientation { x, y, z };

	/**
	 * @brief Axis constructor
	 * @param d orientation, x, y or z (z not implemented), given by enum @c Orientation 
	 * @param xy start point of axis
	 * @param length length of axis
	 * @param nummber_of_notches number of notches distributed along the axis, 0 by default
	 * @param label label to be displayed under axis
	 */
	Axis(Orientation d, Point xy, int length, int nummber_of_notches = 0, string label = "");

	/** @brief Draws the Axis with notches and label  */
	void draw_lines() const;

	/** @brief moves the entire axis with notches and labels @p dx and @p dy in their respective directions  */
	void move(int dx, int dy);

	/** @brief set color of axis, notches and label to c  */
	void set_color(Color c);

	Text label;
	Lines notches;
	//	Orientation orin;
	//	int notches;
};

/** @brief A Circle, inherits from Shape  */
struct Circle : Shape
{
	/** @brief Circle constructor from center in @p p with radius @p rr  */
	Circle(Point p, int rr) : r{rr}
	{
		add(Point{p.x - r, p.y - r});
	}

	/** @brief draws the circle, and fills it if fillcolor is set  */
	void draw_lines() const;

	/** @brief returns center of the circle  */
	Point center() const { return {point(0).x + r, point(0).y + r}; }

	/** @brief change the radius of the circle to rr  */
	void set_radius(int rr) { r = rr; }

	/** @brief returns the radius of the circle  */
	int radius() const { return r; }

private:
	/** @brief radius */
	int r;
};

/** @brief An arced line-segment, inherits from Shape  */
class Arc : public Shape
{
public:
	/**
	 * @brief Arc constructor, creates a part of a circle from @p sd to @p ed around @p center
	 * @param center center of arc
	 * @param w width of circle
	 * @param h height of circle
	 * @param sd start degree
	 * @param ed end degree
	  */
	Arc(Point center, int w, int h, int sd, int ed)
		: w{w}, h{h}, start_deg{sd}, end_deg{ed}
	{
		add(Point{center.x - w / 2, center.y - h / 2});
	}

	/** @brief Draws the arc from @c sd to @c ed  */
	void draw_lines() const override;

	/** @brief Change @c sd to @p d  */
	void set_start(int d) { start_deg = d; }

	/** @brief Change @c ed to @p d  */
	void set_end(int d) { end_deg = d; }

	/** @brief Change width of arc to @p w  */
	void setw(int w) { this->w = w; }

	/** @brief Change height of arc to @p h  */
	void seth(int h) { this->h = h; }

private:
	int w;
	int h;
	int start_deg;
	int end_deg;
};

/** @brief An ellipse (oval), inherits from Shape  */
struct Ellipse : Shape
{
	/**
	 * @brief Ellipse constructor
	 * @param p center
	 * @param ww width, max distance from center
	 * @param hh height, min distance from center
	  */
	Ellipse(Point p, int ww, int hh)
		: w{ww}, h{hh}
	{
		add(Point{p.x - ww, p.y - hh});
	}

	/** @brief Draws the Ellipse and fills it if fillcollor is set  */
	void draw_lines() const;

	/** @brief Returns center of Ellipse  */
	Point center() const { return {point(0).x + w, point(0).y + h}; }
	Point focus1() const { return {center().x + int(sqrt(double(w * w - h * h))), center().y}; }
	Point focus2() const { return {center().x - int(sqrt(double(w * w - h * h))), center().y}; }

	/** @brief set max from center to @p ww  */
	void set_major(int ww) { w = ww; }

	/** @brief returns mex length from center  */
	int major() const { return w; }

	/** @brief set min length from center to @p hh  */
	void set_minor(int hh) { h = hh; }

	/** @brief returns min length from center  */
	int minor() const { return h; }

private:
	int w;
	int h;
};

/** @brief An open polyline with marks for each Point on the line, inherits from Open_polyline  */
struct Marked_polyline : Open_polyline
{
	/** @brief Creates a Marked_polyline with @p m as marks for every Point  */
	Marked_polyline(const string& m) : mark(m) {}

	/** @brief Draws the line and its marks  */
	void draw_lines() const;

private:
	string mark;
};

/** @brief Marks that can be stored as a vector of Points in member-vector @c points inherits from Marked_polyline  */
struct Marks : Marked_polyline
{
	/** @brief Constructs marks, will be displayed as @p m  */
	Marks(const string& m) : Marked_polyline(m)
	{
		set_color(Color(Color::invisible));
	}
};

/** @brief A single Marks, inherits from Marks  */
struct Mark : Marks
{
	/** @brief Constructs a Marks at point @p xy , displayed as @p c  */
	Mark(Point xy, char c) : Marks(string(1, c))
	{
		add(xy);
	}
};

struct Bad_image : Fl_Image
{
	Bad_image(int h, int w) : Fl_Image(h, w, 0)
	{}
	void draw(int x, int y, int, int, int, int)
	{
		draw_empty(x, y);
	}
};

struct Suffix
{
	enum Encoding { none, jpg, gif, bmp };
};

Suffix::Encoding get_encoding(const string& s);

struct Image : Shape
{
	Image(Point xy, string s, Suffix::Encoding e = Suffix::none);
	~Image() { delete p; }
	void draw_lines() const;
	void set_mask(Point xy, int ww, int hh)
	{
		w = ww;
		h = hh;
		cx = xy.x;
		cy = xy.y;
	}
	void move(int dx, int dy)
	{
		Shape::move(dx, dy);
		p->draw(point(0).x, point(0).y);
	}

private:
	int w, h, cx, cy; // define "masking box" within image relative to
					  // position (cx,cy)
	Fl_Image* p;
	Text fn;
};

} // namespace Graph_lib
